const IRREGULAR_FORMS = {
  // 동사
  "studies": "study",
  "studied": "study",
  "studying": "study",
  "tries": "try",
  "tried": "try",
  "trying": "try",
  "lies": "lie",
  "lied": "lie",
  "lying": "lie",
  "dies": "die",
  "died": "die",
  "dying": "die",
  "ties": "tie",
  "tied": "tie",
  "tying": "tie",
  "goes": "go",
  "went": "go",
  "gone": "go",
  "going": "go",
  "does": "do",
  "did": "do",
  "done": "do",
  "doing": "do",
  "has": "have",
  "had": "have",
  "having": "have",
  "is": "be",
  "am": "be",
  "are": "be",
  "was": "be",
  "were": "be",
  "been": "be",
  "being": "be",
  "ran": "run",
  "running": "run",
  "runs": "run",
  "took": "take",
  "taken": "take",
  "taking": "take",
  "made": "make",
  "making": "make",
  "said": "say",
  "saying": "say",
  "got": "get",
  "gotten": "get",
  "getting": "get",
  "came": "come",
  "coming": "come",
  "knew": "know",
  "known": "know",
  "knowing": "know",
  "thought": "think",
  "thinking": "think",
  "saw": "see",
  "seen": "see",
  "seeing": "see",
  "wrote": "write",
  "written": "write",
  "writing": "write",
  "gave": "give",
  "given": "give",
  "giving": "give",
  "found": "find",
  "finding": "find",
  "told": "tell",
  "telling": "tell",
  "felt": "feel",
  "feeling": "feel",
  "became": "become",
  "becoming": "become",
  "left": "leave",
  "leaving": "leave",
  "kept": "keep",
  "keeping": "keep",
  "let": "let",
  "letting": "let",
  "began": "begin",
  "begun": "begin",
  "beginning": "begin",
  "seemed": "seem",
  "seeming": "seem",
  "helped": "help",
  "helping": "help",
  "showed": "show",
  "shown": "show",
  "showing": "show",
  "heard": "hear",
  "hearing": "hear",
  "played": "play",
  "playing": "play",
  "moved": "move",
  "moving": "move",
  "lived": "live",
  "living": "live",
  "believed": "believe",
  "believing": "believe",
  "brought": "bring",
  "bringing": "bring",
  "happened": "happen",
  "happening": "happen",
  "held": "hold",
  "holding": "hold",
  "stood": "stand",
  "standing": "stand",
  "understood": "understand",
  "understanding": "understand",
  // 명사
  "men": "man",
  "women": "woman",
  "children": "child",
  "feet": "foot",
  "teeth": "tooth",
  "mice": "mouse",
  "geese": "goose",
  "people": "person",
  "phenomena": "phenomenon",
  "criteria": "criterion",
  "data": "datum",
  "analyses": "analysis",
  "theses": "thesis",
  "hypotheses": "hypothesis",
  "crises": "crisis",
  "bases": "basis"
};
const DOUBLE_CONSONANT_ING = /^(.+)(bb|dd|gg|ll|mm|nn|pp|rr|ss|tt|zz)ing$/;
const DOUBLE_CONSONANT_ED = /^(.+)(bb|dd|gg|ll|mm|nn|pp|rr|ss|tt|zz)ed$/;
function toLemma(word) {
  const normalized = word.toLowerCase().trim();
  if (!normalized) {
    return normalized;
  }
  if (IRREGULAR_FORMS[normalized]) {
    return IRREGULAR_FORMS[normalized];
  }
  if (normalized.length <= 3) {
    return normalized;
  }
  if (normalized.endsWith("ing")) {
    const doubleMatch = normalized.match(DOUBLE_CONSONANT_ING);
    if (doubleMatch) {
      return doubleMatch[1] + doubleMatch[2][0];
    }
    if (normalized.endsWith("ying")) {
      return normalized.slice(0, -4) + "y";
    }
    const withoutIng = normalized.slice(0, -3);
    if (withoutIng.length >= 2) {
      return withoutIng + "e";
    }
    return withoutIng;
  }
  if (normalized.endsWith("ed")) {
    const doubleMatch = normalized.match(DOUBLE_CONSONANT_ED);
    if (doubleMatch) {
      return doubleMatch[1] + doubleMatch[2][0];
    }
    if (normalized.endsWith("ied")) {
      return normalized.slice(0, -3) + "y";
    }
    const withoutEd = normalized.slice(0, -2);
    if (!normalized.endsWith("eed")) {
      return withoutEd;
    }
    return withoutEd;
  }
  if (normalized.endsWith("es")) {
    if (normalized.endsWith("ies")) {
      return normalized.slice(0, -3) + "y";
    }
    if (/(?:sh|ch|x|ss|zz)es$/.test(normalized)) {
      return normalized.slice(0, -2);
    }
    if (normalized.endsWith("oes")) {
      return normalized.slice(0, -2);
    }
    return normalized.slice(0, -2);
  }
  if (normalized.endsWith("s") && !normalized.endsWith("ss")) {
    return normalized.slice(0, -1);
  }
  if (normalized.endsWith("er") && normalized.length > 4) {
    if (normalized.endsWith("ier")) {
      return normalized.slice(0, -3) + "y";
    }
    const withoutEr = normalized.slice(0, -2);
    if (/(.)\1$/.test(withoutEr)) {
      return withoutEr.slice(0, -1);
    }
    return withoutEr;
  }
  if (normalized.endsWith("est") && normalized.length > 5) {
    if (normalized.endsWith("iest")) {
      return normalized.slice(0, -4) + "y";
    }
    const withoutEst = normalized.slice(0, -3);
    if (/(.)\1$/.test(withoutEst)) {
      return withoutEst.slice(0, -1);
    }
    return withoutEst;
  }
  if (normalized.endsWith("ly") && normalized.length > 4) {
    if (normalized.endsWith("ily")) {
      return normalized.slice(0, -3) + "y";
    }
    if (normalized.endsWith("ally") && normalized.length > 6) {
      return normalized.slice(0, -2);
    }
    return normalized.slice(0, -2);
  }
  return normalized;
}
function getLemmaCandidates(word) {
  const normalized = word.toLowerCase().trim();
  const primary = toLemma(normalized);
  const candidates = /* @__PURE__ */ new Set();
  candidates.add(normalized);
  candidates.add(primary);
  if (IRREGULAR_FORMS[normalized]) {
    candidates.add(IRREGULAR_FORMS[normalized]);
  }
  return Array.from(candidates);
}
function isEnglishWord(text) {
  if (text.length < 2) {
    return false;
  }
  return /^[a-zA-Z]+(?:[-'][a-zA-Z]+)*$/.test(text);
}

const DEEPL_API_FREE = "https://api-free.deepl.com/v2/translate";
const DEEPL_API_PRO = "https://api.deepl.com/v2/translate";
const CONFIG_KEY = "engeagle_deepl_config";
async function saveDeepLConfig(config) {
  await chrome.storage.local.set({ [CONFIG_KEY]: config });
}
async function loadDeepLConfig() {
  const result = await chrome.storage.local.get(CONFIG_KEY);
  return result[CONFIG_KEY] || null;
}
function isValidApiKey(apiKey) {
  return apiKey.length > 10 && /^[a-zA-Z0-9\-:]+$/.test(apiKey);
}
function isFreeApiKey(apiKey) {
  return apiKey.endsWith(":fx");
}
async function translateWithDeepL(word, config) {
  if (!config.apiKey || !isValidApiKey(config.apiKey)) {
    console.warn("[EngEagle] Invalid DeepL API key");
    return null;
  }
  const endpoint = config.useFreeApi ? DEEPL_API_FREE : DEEPL_API_PRO;
  try {
    const response = await fetch(endpoint, {
      method: "POST",
      headers: {
        "Authorization": `DeepL-Auth-Key ${config.apiKey}`,
        "Content-Type": "application/json",
        "User-Agent": "EngEagle/1.0.0"
      },
      body: JSON.stringify({
        text: [word],
        target_lang: "KO",
        source_lang: "EN"
      })
    });
    if (!response.ok) {
      if (response.status === 403) {
        console.error("[EngEagle] DeepL API key invalid or quota exceeded");
      } else if (response.status === 456) {
        console.error("[EngEagle] DeepL quota exceeded");
      } else {
        console.error(`[EngEagle] DeepL API error: ${response.status}`);
      }
      return null;
    }
    const data = await response.json();
    if (data.translations && data.translations.length > 0) {
      const translation = data.translations[0];
      return {
        word,
        meanings: [translation.text],
        detectedSourceLang: translation.detected_source_language || "EN"
      };
    }
    return null;
  } catch (error) {
    console.error("[EngEagle] DeepL API request failed:", error);
    return null;
  }
}
async function testDeepLConnection(config) {
  if (!config.apiKey || !isValidApiKey(config.apiKey)) {
    return { success: false, message: "API 키 형식이 올바르지 않습니다." };
  }
  const endpoint = config.useFreeApi ? "https://api-free.deepl.com/v2/usage" : "https://api.deepl.com/v2/usage";
  try {
    const response = await fetch(endpoint, {
      method: "GET",
      headers: {
        "Authorization": `DeepL-Auth-Key ${config.apiKey}`,
        "User-Agent": "EngEagle/1.0.0"
      }
    });
    if (!response.ok) {
      if (response.status === 403) {
        return { success: false, message: "API 키가 유효하지 않습니다." };
      }
      return { success: false, message: `API 오류: ${response.status}` };
    }
    const data = await response.json();
    return {
      success: true,
      message: "연결 성공!",
      usage: {
        character_count: data.character_count,
        character_limit: data.character_limit
      }
    };
  } catch (error) {
    return { success: false, message: "네트워크 오류가 발생했습니다." };
  }
}
function deepLResultToDictEntry(word, translation) {
  return {
    word,
    lemma: word.toLowerCase(),
    pos: "",
    // DeepL은 품사 정보를 제공하지 않음 (UI에서 빈 값 처리)
    meanings: translation.meanings,
    example: "",
    source_url: "deepl"
  };
}

const posMap = {
  "noun": "n",
  "verb": "v",
  "adjective": "adj",
  "adverb": "adv",
  "pronoun": "pron",
  "preposition": "prep",
  "conjunction": "conj",
  "interjection": "interj",
  "determiner": "det",
  "exclamation": "excl"
};
async function lookupFreeDictionary(word) {
  const normalizedWord = word.toLowerCase().trim();
  try {
    const response = await fetch(
      `https://api.dictionaryapi.dev/api/v2/entries/en/${encodeURIComponent(normalizedWord)}`,
      {
        method: "GET",
        headers: {
          "Accept": "application/json"
        }
      }
    );
    if (!response.ok) {
      if (response.status === 404) {
        console.log(`[EngEagle] Free Dictionary: Word not found - ${normalizedWord}`);
        return null;
      }
      throw new Error(`Free Dictionary API error: ${response.status}`);
    }
    const data = await response.json();
    if (!data || data.length === 0) {
      return null;
    }
    return freeDictToEntry(data[0]);
  } catch (error) {
    console.error("[EngEagle] Free Dictionary API error:", error);
    return null;
  }
}
function freeDictToEntry(data) {
  const meanings = [];
  let pos = "";
  let example = "";
  for (const meaning of data.meanings) {
    if (!pos && meaning.partOfSpeech) {
      pos = posMap[meaning.partOfSpeech.toLowerCase()] || meaning.partOfSpeech;
    }
    for (const def of meaning.definitions) {
      if (meanings.length >= 5) break;
      meanings.push(def.definition);
      if (!example && def.example) {
        example = def.example;
      }
    }
    if (meanings.length >= 5) break;
  }
  return {
    word: data.word,
    lemma: data.word,
    pos,
    meanings,
    example,
    source_url: `https://dictionaryapi.dev/`
  };
}
async function enrichWithKoreanTranslation(entry, translateFn) {
  try {
    if (entry.meanings.length > 0) {
      const koreanMeaning = await translateFn(entry.meanings[0]);
      if (koreanMeaning) {
        entry.meanings = [
          `${koreanMeaning}`,
          ...entry.meanings.slice(0, 2).map((m) => `📖 ${m}`)
        ];
      }
    }
  } catch (error) {
    console.error("[EngEagle] Translation enrichment error:", error);
  }
  return entry;
}

class LRUCache {
  cache;
  maxSize;
  constructor(maxSize = 500) {
    this.cache = /* @__PURE__ */ new Map();
    this.maxSize = maxSize;
  }
  get(key) {
    if (!this.cache.has(key)) {
      return void 0;
    }
    const value = this.cache.get(key);
    this.cache.delete(key);
    this.cache.set(key, value);
    return value;
  }
  set(key, value) {
    if (this.cache.has(key)) {
      this.cache.delete(key);
    } else if (this.cache.size >= this.maxSize) {
      const firstKey = this.cache.keys().next().value;
      if (firstKey !== void 0) {
        this.cache.delete(firstKey);
      }
    }
    this.cache.set(key, value);
  }
  has(key) {
    return this.cache.has(key);
  }
  clear() {
    this.cache.clear();
  }
  get size() {
    return this.cache.size;
  }
}
let dictIndex = null;
let isLoading = false;
let loadPromise = null;
const lookupCache = new LRUCache(500);
const failedLookupCache = /* @__PURE__ */ new Map();
const FAILED_CACHE_TTL = 5 * 60 * 1e3;
async function loadDictionary() {
  if (dictIndex !== null) {
    return;
  }
  if (isLoading && loadPromise) {
    return loadPromise;
  }
  isLoading = true;
  loadPromise = (async () => {
    try {
      const url = chrome.runtime.getURL("assets/dict_en_ko_min.json");
      const response = await fetch(url);
      if (!response.ok) {
        throw new Error(`Failed to load dictionary: ${response.status}`);
      }
      const data = await response.json();
      dictIndex = /* @__PURE__ */ new Map();
      for (const [key, entry] of Object.entries(data)) {
        dictIndex.set(key.toLowerCase(), entry);
        if (entry.word.toLowerCase() !== key.toLowerCase()) {
          if (!dictIndex.has(entry.word.toLowerCase())) {
            dictIndex.set(entry.word.toLowerCase(), entry);
          }
        }
      }
      console.log(`[EngEagle] Dictionary loaded: ${dictIndex.size} entries`);
    } catch (error) {
      console.error("[EngEagle] Failed to load dictionary:", error);
      dictIndex = /* @__PURE__ */ new Map();
    } finally {
      isLoading = false;
    }
  })();
  return loadPromise;
}
async function lookupWordLocal(word) {
  const startTime = performance.now();
  const normalizedWord = word.toLowerCase().trim();
  if (dictIndex === null) {
    await loadDictionary();
  }
  if (!dictIndex) {
    const lookupTime2 = performance.now() - startTime;
    return { found: false, cached: false, lookupTime: lookupTime2, source: "local" };
  }
  const candidates = getLemmaCandidates(normalizedWord);
  for (const candidate of candidates) {
    const entry = dictIndex.get(candidate);
    if (entry) {
      const lookupTime2 = performance.now() - startTime;
      return {
        found: true,
        entry,
        cached: false,
        lookupTime: lookupTime2,
        source: "local"
      };
    }
  }
  const lookupTime = performance.now() - startTime;
  return { found: false, cached: false, lookupTime, source: "local" };
}
async function lookupWord(word) {
  const startTime = performance.now();
  const normalizedWord = word.toLowerCase().trim();
  if (lookupCache.has(normalizedWord)) {
    const cached = lookupCache.get(normalizedWord);
    const lookupTime2 = performance.now() - startTime;
    if (cached) {
      return {
        found: true,
        entry: cached,
        cached: true,
        lookupTime: lookupTime2,
        source: "cache"
      };
    } else {
      return {
        found: false,
        cached: true,
        lookupTime: lookupTime2,
        source: "cache"
      };
    }
  }
  const localResult = await lookupWordLocal(normalizedWord);
  if (localResult.found && localResult.entry) {
    lookupCache.set(normalizedWord, localResult.entry);
    const lookupTime2 = performance.now() - startTime;
    return {
      found: true,
      entry: localResult.entry,
      cached: false,
      lookupTime: lookupTime2,
      source: "local"
    };
  }
  const lastFailed = failedLookupCache.get(normalizedWord);
  const shouldTryOnline = !lastFailed || Date.now() - lastFailed > FAILED_CACHE_TTL;
  if (shouldTryOnline) {
    const freeDictResult = await lookupWithFreeDictionary(word);
    if (freeDictResult) {
      failedLookupCache.delete(normalizedWord);
      lookupCache.set(normalizedWord, freeDictResult);
      const lookupTime2 = performance.now() - startTime;
      return {
        found: true,
        entry: freeDictResult,
        cached: false,
        lookupTime: lookupTime2,
        source: "freedict"
      };
    }
    const deepLResult = await lookupWithDeepL(word);
    if (deepLResult) {
      failedLookupCache.delete(normalizedWord);
      lookupCache.set(normalizedWord, deepLResult);
      const lookupTime2 = performance.now() - startTime;
      return {
        found: true,
        entry: deepLResult,
        cached: false,
        lookupTime: lookupTime2,
        source: "deepl"
      };
    }
    failedLookupCache.set(normalizedWord, Date.now());
  }
  const lookupTime = performance.now() - startTime;
  return {
    found: false,
    cached: false,
    lookupTime
  };
}
async function lookupWithFreeDictionary(word) {
  try {
    console.log(`[EngEagle] Looking up "${word}" with Free Dictionary...`);
    const entry = await lookupFreeDictionary(word);
    if (!entry) {
      return null;
    }
    console.log(`[EngEagle] Free Dictionary found: ${entry.meanings[0]?.substring(0, 50)}...`);
    const config = await loadDeepLConfig();
    if (config?.apiKey && entry.meanings.length > 0) {
      const enriched = await enrichWithKoreanTranslation(entry, async (text) => {
        const translation = await translateWithDeepL(text, config);
        return translation?.meanings[0] || null;
      });
      return enriched;
    }
    return entry;
  } catch (error) {
    console.error("[EngEagle] Free Dictionary lookup failed:", error);
    return null;
  }
}
async function lookupWithDeepL(word) {
  try {
    const config = await loadDeepLConfig();
    if (!config || !config.apiKey) {
      console.log("[EngEagle] DeepL not configured, skipping");
      return null;
    }
    console.log(`[EngEagle] Looking up "${word}" with DeepL...`);
    const translation = await translateWithDeepL(word, config);
    if (translation) {
      console.log(`[EngEagle] DeepL translation: ${translation.meanings.join(", ")}`);
      return deepLResultToDictEntry(word, translation);
    }
    return null;
  } catch (error) {
    console.error("[EngEagle] DeepL lookup failed:", error);
    return null;
  }
}
function clearCache() {
  lookupCache.clear();
  failedLookupCache.clear();
}
function isDictionaryLoaded() {
  return dictIndex !== null && dictIndex.size > 0;
}
function getDictionarySize() {
  return dictIndex?.size ?? 0;
}

function djb2Hash(str) {
  let hash = 5381;
  for (let i = 0; i < str.length; i++) {
    hash = (hash << 5) + hash + str.charCodeAt(i);
    hash = hash & 4294967295;
  }
  return hash >>> 0;
}
function generateWordId(lemma, lang = "en") {
  const input = `${lemma.toLowerCase()}|${lang}`;
  const hash = djb2Hash(input);
  return hash.toString(16).padStart(8, "0");
}

const DB_NAME = "engeagle_db";
const DB_VERSION = 1;
const STORE_NAME = "vocabulary";
const retryQueue = [];
let isProcessingRetry = false;
function openDatabase() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open(DB_NAME, DB_VERSION);
    request.onerror = () => reject(request.error);
    request.onsuccess = () => resolve(request.result);
    request.onupgradeneeded = (event) => {
      const db = event.target.result;
      if (!db.objectStoreNames.contains(STORE_NAME)) {
        const store = db.createObjectStore(STORE_NAME, { keyPath: "id" });
        store.createIndex("lemma", "lemma", { unique: false });
        store.createIndex("created_at", "created_at", { unique: false });
        store.createIndex("freq", "freq", { unique: false });
      }
    };
  });
}
async function saveWord(entry) {
  try {
    const id = generateWordId(entry.lemma);
    const db = await openDatabase();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction(STORE_NAME, "readwrite");
      const store = transaction.objectStore(STORE_NAME);
      const getRequest = store.get(id);
      getRequest.onsuccess = () => {
        const existing = getRequest.result;
        let newEntry;
        let isNew = true;
        if (existing) {
          isNew = false;
          newEntry = {
            ...existing,
            freq: existing.freq + 1
            // source_url은 첫 번째 것 유지
          };
        } else {
          newEntry = {
            id,
            word: entry.word,
            lemma: entry.lemma,
            pos: entry.pos,
            meanings: entry.meanings,
            example: entry.example,
            source_url: entry.source_url,
            created_at: Math.floor(Date.now() / 1e3),
            freq: 1
          };
        }
        const putRequest = store.put(newEntry);
        putRequest.onsuccess = () => {
          resolve({ success: true, isNew, entry: newEntry });
        };
        putRequest.onerror = () => {
          addToRetryQueue(newEntry);
          reject(putRequest.error);
        };
      };
      getRequest.onerror = () => {
        reject(getRequest.error);
      };
      transaction.oncomplete = () => db.close();
    });
  } catch (error) {
    console.error("[EngEagle] Save failed:", error);
    throw error;
  }
}
async function getAllWords() {
  try {
    const db = await openDatabase();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction(STORE_NAME, "readonly");
      const store = transaction.objectStore(STORE_NAME);
      const request = store.getAll();
      request.onsuccess = () => resolve(request.result || []);
      request.onerror = () => reject(request.error);
      transaction.oncomplete = () => db.close();
    });
  } catch (error) {
    console.error("[EngEagle] GetAll failed:", error);
    return [];
  }
}
async function deleteWord(id) {
  try {
    const db = await openDatabase();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction(STORE_NAME, "readwrite");
      const store = transaction.objectStore(STORE_NAME);
      const request = store.delete(id);
      request.onsuccess = () => resolve(true);
      request.onerror = () => reject(request.error);
      transaction.oncomplete = () => db.close();
    });
  } catch (error) {
    console.error("[EngEagle] Delete failed:", error);
    return false;
  }
}
async function restoreWord(entry) {
  try {
    const db = await openDatabase();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction(STORE_NAME, "readwrite");
      const store = transaction.objectStore(STORE_NAME);
      const request = store.put(entry);
      request.onsuccess = () => resolve(true);
      request.onerror = () => reject(request.error);
      transaction.oncomplete = () => db.close();
    });
  } catch (error) {
    console.error("[EngEagle] Restore failed:", error);
    return false;
  }
}
async function updateWord(id, updates) {
  try {
    const db = await openDatabase();
    return new Promise((resolve, reject) => {
      const transaction = db.transaction(STORE_NAME, "readwrite");
      const store = transaction.objectStore(STORE_NAME);
      const getRequest = store.get(id);
      getRequest.onsuccess = () => {
        const existing = getRequest.result;
        if (!existing) {
          resolve(null);
          return;
        }
        const updatedEntry = {
          ...existing,
          ...updates
        };
        const putRequest = store.put(updatedEntry);
        putRequest.onsuccess = () => resolve(updatedEntry);
        putRequest.onerror = () => reject(putRequest.error);
      };
      getRequest.onerror = () => reject(getRequest.error);
      transaction.oncomplete = () => db.close();
    });
  } catch (error) {
    console.error("[EngEagle] Update failed:", error);
    return null;
  }
}
async function searchWords(prefix) {
  try {
    const allWords = await getAllWords();
    const lowerPrefix = prefix.toLowerCase();
    return allWords.filter(
      (entry) => entry.word.toLowerCase().startsWith(lowerPrefix) || entry.lemma.toLowerCase().startsWith(lowerPrefix)
    );
  } catch (error) {
    console.error("[EngEagle] Search failed:", error);
    return [];
  }
}
async function exportToJSON() {
  const words = await getAllWords();
  return JSON.stringify(words, null, 2);
}
async function exportToCSV() {
  const words = await getAllWords();
  const headers = ["word", "lemma", "pos", "meanings", "example", "freq", "created_at"];
  const rows = words.map((entry) => [
    entry.word,
    entry.lemma,
    entry.pos,
    entry.meanings.join("; "),
    entry.example,
    entry.freq.toString(),
    new Date(entry.created_at * 1e3).toISOString()
  ].map((cell) => `"${cell.replace(/"/g, '""')}"`).join(","));
  return [headers.join(","), ...rows].join("\n");
}
async function importFromJSON(jsonString) {
  try {
    const entries = JSON.parse(jsonString);
    let imported = 0;
    for (const entry of entries) {
      if (entry.id && entry.word && entry.lemma) {
        await restoreWord(entry);
        imported++;
      }
    }
    return imported;
  } catch (error) {
    console.error("[EngEagle] Import failed:", error);
    return 0;
  }
}
function addToRetryQueue(entry) {
  retryQueue.push(entry);
  processRetryQueue();
}
async function processRetryQueue() {
  if (isProcessingRetry || retryQueue.length === 0) return;
  isProcessingRetry = true;
  while (retryQueue.length > 0) {
    const entry = retryQueue.shift();
    try {
      await restoreWord(entry);
    } catch {
      if (!entry.hasOwnProperty("_retryCount")) {
        entry._retryCount = 1;
      } else if (entry._retryCount < 3) {
        entry._retryCount++;
        retryQueue.push(entry);
      }
    }
    await new Promise((resolve) => setTimeout(resolve, 1e3));
  }
  isProcessingRetry = false;
}
async function getWordCount() {
  const words = await getAllWords();
  return words.length;
}

const TRIGGER_CONFIG_KEY = "engeagle_trigger_config";
const DEFAULT_TRIGGER_CONFIG = {
  altDblclick: false,
  // Alt/Option + 더블클릭
  shiftDblclick: true,
  // Shift + 더블클릭 (기본)
  contextMenu: true
};
chrome.runtime.onInstalled.addListener(async () => {
  console.log("[EngEagle] Extension installed/updated");
  await loadDictionary();
  console.log(`[EngEagle] Dictionary ready: ${getDictionarySize()} entries`);
  await setupContextMenu();
});
async function setupContextMenu() {
  await chrome.contextMenus.removeAll();
  const config = await loadTriggerConfig();
  if (config.contextMenu) {
    chrome.contextMenus.create({
      id: "engeagle-translate",
      title: "EngEagle로 번역",
      contexts: ["selection"]
    });
    console.log("[EngEagle] Context menu created");
  }
}
async function loadTriggerConfig() {
  try {
    const result = await chrome.storage.local.get(TRIGGER_CONFIG_KEY);
    const saved = result[TRIGGER_CONFIG_KEY];
    if (saved) {
      return { ...DEFAULT_TRIGGER_CONFIG, ...saved };
    }
    return DEFAULT_TRIGGER_CONFIG;
  } catch {
    return DEFAULT_TRIGGER_CONFIG;
  }
}
async function saveTriggerConfig(config) {
  await chrome.storage.local.set({ [TRIGGER_CONFIG_KEY]: config });
  await setupContextMenu();
}
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  console.log("[EngEagle] Context menu clicked:", info.menuItemId, info.selectionText);
  if (info.menuItemId === "engeagle-translate" && info.selectionText && tab?.id) {
    const word = info.selectionText.trim().split(/\s+/)[0];
    console.log("[EngEagle] Looking up word:", word);
    if (isEnglishWord(word)) {
      try {
        const result = await lookupWord(word);
        console.log("[EngEagle] Lookup result:", result);
        try {
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            files: ["src/content/selection.js"]
          });
        } catch {
        }
        try {
          await chrome.tabs.sendMessage(tab.id, {
            type: "SHOW_TRANSLATION",
            word,
            result,
            sourceUrl: tab.url || ""
          });
          console.log("[EngEagle] Message sent to tab");
        } catch (sendError) {
          console.warn("[EngEagle] Could not send message to tab (page may not support content scripts):", sendError);
        }
      } catch (error) {
        console.error("[EngEagle] Context menu translation error:", error);
      }
    } else {
      console.log("[EngEagle] Not an English word:", word);
    }
  }
});
(async () => {
  await loadDictionary();
  console.log("[EngEagle] Service worker started, dictionary loaded");
})();
chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  handleMessage(message).then(sendResponse).catch((error) => {
    console.error("[EngEagle] Message handling error:", error);
    sendResponse({ success: false, error: error.message });
  });
  return true;
});
async function handleMessage(message) {
  switch (message.type) {
    case "LOOKUP":
      return handleLookup(message);
    case "SAVE":
      return handleSave(message);
    case "GET_ALL":
      return handleGetAll();
    case "DELETE":
      return handleDelete(message);
    case "RESTORE":
      return handleRestore(message);
    case "UPDATE":
      return handleUpdate(message);
    case "SEARCH":
      return handleSearch(message);
    case "EXPORT":
      return handleExport(message);
    case "IMPORT":
      return handleImport(message);
    case "STATUS":
      return handleStatus();
    case "DEEPL_SAVE_CONFIG":
      return handleDeepLSaveConfig(message);
    case "DEEPL_LOAD_CONFIG":
      return handleDeepLLoadConfig();
    case "DEEPL_TEST":
      return handleDeepLTest(message);
    case "TRIGGER_SAVE_CONFIG":
      return handleTriggerSaveConfig(message);
    case "TRIGGER_LOAD_CONFIG":
      return handleTriggerLoadConfig();
    default:
      return { success: false, error: "Unknown message type" };
  }
}
async function handleLookup(message) {
  const { word, sourceUrl = "", saveToVocabulary = false } = message;
  if (!isEnglishWord(word)) {
    return {
      success: false,
      error: "Not a valid English word"
    };
  }
  const result = await lookupWord(word);
  if (!result.found || !result.entry) {
    return {
      success: false,
      error: "Word not found",
      lookupTime: result.lookupTime
    };
  }
  if (saveToVocabulary) {
    saveWordAsync(result.entry, sourceUrl);
  }
  return {
    success: true,
    entry: result.entry,
    cached: result.cached,
    lookupTime: result.lookupTime,
    source: result.source
    // 'local', 'deepl', 'cache'
  };
}
function saveWordAsync(entry, sourceUrl) {
  saveWord({
    word: entry.word,
    lemma: entry.lemma,
    pos: entry.pos,
    meanings: entry.meanings,
    example: entry.example,
    source_url: sourceUrl || entry.source_url
  }).then((result) => {
    console.log(`[EngEagle] Word saved: ${entry.word} (new: ${result.isNew}, freq: ${result.entry.freq})`);
  }).catch((error) => {
    console.error("[EngEagle] Save failed:", error);
  });
}
async function handleSave(message) {
  try {
    const result = await saveWord(message.entry);
    return {
      success: true,
      isNew: result.isNew,
      entry: result.entry
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleGetAll() {
  try {
    const entries = await getAllWords();
    return {
      success: true,
      entries,
      count: entries.length
    };
  } catch (error) {
    return {
      success: false,
      error: error.message,
      entries: []
    };
  }
}
async function handleDelete(message) {
  try {
    const success = await deleteWord(message.id);
    return { success };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleRestore(message) {
  try {
    const success = await restoreWord(message.entry);
    return { success };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleUpdate(message) {
  try {
    const entry = await updateWord(message.id, message.updates);
    if (entry) {
      return { success: true, entry };
    }
    return { success: false, error: "Word not found" };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleSearch(message) {
  try {
    const entries = await searchWords(message.prefix);
    return {
      success: true,
      entries,
      count: entries.length
    };
  } catch (error) {
    return {
      success: false,
      error: error.message,
      entries: []
    };
  }
}
async function handleExport(message) {
  try {
    const data = message.format === "csv" ? await exportToCSV() : await exportToJSON();
    return {
      success: true,
      data,
      format: message.format
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleImport(message) {
  try {
    const count = await importFromJSON(message.data);
    return {
      success: true,
      imported: count
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleStatus() {
  try {
    const wordCount = await getWordCount();
    const deepLConfig = await loadDeepLConfig();
    return {
      success: true,
      dictionaryLoaded: isDictionaryLoaded(),
      dictionarySize: getDictionarySize(),
      vocabularyCount: wordCount,
      deepLConfigured: !!(deepLConfig && deepLConfig.apiKey)
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleDeepLSaveConfig(message) {
  try {
    const config = {
      ...message.config,
      useFreeApi: isFreeApiKey(message.config.apiKey)
    };
    await saveDeepLConfig(config);
    clearCache();
    console.log("[EngEagle] DeepL config saved, cache cleared");
    return { success: true };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleDeepLLoadConfig() {
  try {
    const config = await loadDeepLConfig();
    return {
      success: true,
      config
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleDeepLTest(message) {
  try {
    const config = {
      ...message.config,
      useFreeApi: isFreeApiKey(message.config.apiKey)
    };
    const result = await testDeepLConnection(config);
    return {
      success: result.success,
      message: result.message,
      usage: result.usage
    };
  } catch (error) {
    return {
      success: false,
      message: error.message
    };
  }
}
async function handleTriggerSaveConfig(message) {
  try {
    await saveTriggerConfig(message.config);
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      if (tab.id) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            type: "TRIGGER_CONFIG_CHANGED",
            config: message.config
          });
        } catch {
        }
      }
    }
    return { success: true };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
async function handleTriggerLoadConfig() {
  try {
    const config = await loadTriggerConfig();
    return { success: true, config };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}
