"use strict";
(() => {
  // src/lib/lemma.ts
  function isEnglishWord(text) {
    if (text.length < 2) {
      return false;
    }
    return /^[a-zA-Z]+(?:[-'][a-zA-Z]+)*$/.test(text);
  }

  // src/content/selection.ts
  var POPUP_ID = "engeagle-popup";
  var TOAST_ID = "engeagle-toast";
  var currentPopup = null;
  var currentToast = null;
  var hideTimeout = null;
  var scrollHandler = null;
  var resizeHandler = null;
  var currentRange = null;
  function injectStyles() {
    if (document.getElementById("engeagle-styles")) {
      return;
    }
    const style = document.createElement("style");
    style.id = "engeagle-styles";
    style.textContent = `
    #${POPUP_ID} {
      position: absolute;
      z-index: 2147483647;
      max-width: 320px;
      min-width: 200px;
      background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
      border: 1px solid #0f3460;
      border-radius: 12px;
      box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4), 0 0 0 1px rgba(255, 255, 255, 0.05);
      font-family: 'Pretendard', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 14px;
      line-height: 1.5;
      color: #e8e8e8;
      padding: 16px;
      animation: engeagle-fade-in 0.15s ease-out;
      pointer-events: auto;
    }

    @keyframes engeagle-fade-in {
      from {
        opacity: 0;
        transform: translateY(-8px) scale(0.96);
      }
      to {
        opacity: 1;
        transform: translateY(0) scale(1);
      }
    }

    #${POPUP_ID} .engeagle-header {
      display: flex;
      align-items: baseline;
      gap: 8px;
      margin-bottom: 8px;
    }

    #${POPUP_ID} .engeagle-word {
      font-size: 18px;
      font-weight: 700;
      color: #00d9ff;
      letter-spacing: -0.02em;
    }

    #${POPUP_ID} .engeagle-pos {
      font-size: 12px;
      color: #ff6b9d;
      background: rgba(255, 107, 157, 0.15);
      padding: 2px 8px;
      border-radius: 4px;
      font-weight: 500;
    }

    #${POPUP_ID} .engeagle-meanings {
      color: #ffffff;
      font-size: 15px;
      font-weight: 500;
      margin-bottom: 10px;
    }

    #${POPUP_ID} .engeagle-example {
      font-size: 13px;
      color: #a0a0a0;
      font-style: italic;
      padding: 10px 12px;
      background: rgba(255, 255, 255, 0.03);
      border-left: 3px solid #0f3460;
      border-radius: 0 6px 6px 0;
      margin-top: 8px;
    }

    #${POPUP_ID} .engeagle-close {
      position: absolute;
      top: 8px;
      right: 8px;
      width: 24px;
      height: 24px;
      border: none;
      background: transparent;
      color: #666;
      cursor: pointer;
      font-size: 18px;
      line-height: 1;
      border-radius: 4px;
      transition: all 0.15s ease;
    }

    #${POPUP_ID} .engeagle-close:hover {
      background: rgba(255, 255, 255, 0.1);
      color: #fff;
    }

    #${POPUP_ID} .engeagle-error {
      color: #ff6b6b;
      text-align: center;
      padding: 8px;
    }

    #${POPUP_ID} .engeagle-loading {
      text-align: center;
      color: #888;
    }

    #${POPUP_ID} .engeagle-save-btn {
      display: block;
      width: 100%;
      margin-top: 12px;
      padding: 10px 16px;
      background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
      color: white;
      border: none;
      border-radius: 8px;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
      transition: all 0.2s ease;
    }

    #${POPUP_ID} .engeagle-save-btn:hover {
      background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
      transform: translateY(-1px);
    }

    #${POPUP_ID} .engeagle-save-btn:disabled {
      cursor: default;
      transform: none;
    }

    #${TOAST_ID} {
      position: fixed;
      bottom: 24px;
      right: 24px;
      z-index: 2147483647;
      background: linear-gradient(135deg, #00c851 0%, #00a040 100%);
      color: white;
      padding: 12px 20px;
      border-radius: 8px;
      font-family: 'Pretendard', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 14px;
      font-weight: 500;
      box-shadow: 0 4px 16px rgba(0, 200, 81, 0.3);
      animation: engeagle-toast-in 0.3s ease-out;
      display: flex;
      align-items: center;
      gap: 8px;
    }

    @keyframes engeagle-toast-in {
      from {
        opacity: 0;
        transform: translateY(16px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    #${TOAST_ID}.engeagle-toast-out {
      animation: engeagle-toast-out 0.2s ease-in forwards;
    }

    @keyframes engeagle-toast-out {
      from {
        opacity: 1;
        transform: translateY(0);
      }
      to {
        opacity: 0;
        transform: translateY(16px);
      }
    }

    #${TOAST_ID}::before {
      content: '\u2713';
      font-size: 16px;
    }
  `;
    document.head.appendChild(style);
  }
  function hidePopup() {
    if (currentPopup) {
      currentPopup.remove();
      currentPopup = null;
    }
    if (hideTimeout) {
      clearTimeout(hideTimeout);
      hideTimeout = null;
    }
    if (scrollHandler) {
      window.removeEventListener("scroll", scrollHandler, true);
      scrollHandler = null;
    }
    if (resizeHandler) {
      window.removeEventListener("resize", resizeHandler);
      resizeHandler = null;
    }
    currentRange = null;
  }
  function showToast(message) {
    if (currentToast) {
      currentToast.remove();
    }
    const toast = document.createElement("div");
    toast.id = TOAST_ID;
    toast.textContent = message;
    document.body.appendChild(toast);
    currentToast = toast;
    setTimeout(() => {
      if (currentToast === toast) {
        toast.classList.add("engeagle-toast-out");
        setTimeout(() => {
          if (toast.parentNode) {
            toast.remove();
          }
          if (currentToast === toast) {
            currentToast = null;
          }
        }, 200);
      }
    }, 2e3);
  }
  function calculatePopupPosition(range) {
    const rect = range.getBoundingClientRect();
    const scrollX = window.scrollX || document.documentElement.scrollLeft;
    const scrollY = window.scrollY || document.documentElement.scrollTop;
    let top = rect.bottom + scrollY + 8;
    let left = rect.left + scrollX;
    const popupWidth = 320;
    if (left + popupWidth > window.innerWidth + scrollX - 16) {
      left = window.innerWidth + scrollX - popupWidth - 16;
    }
    if (left < scrollX + 16) {
      left = scrollX + 16;
    }
    const estimatedHeight = 150;
    if (rect.bottom + estimatedHeight > window.innerHeight) {
      top = rect.top + scrollY - estimatedHeight - 8;
    }
    return { top, left };
  }
  function updatePopupPosition() {
    if (!currentPopup || !currentRange) return;
    const { top, left } = calculatePopupPosition(currentRange);
    currentPopup.style.top = `${top}px`;
    currentPopup.style.left = `${left}px`;
  }
  function showPopup(range, content) {
    hidePopup();
    injectStyles();
    currentRange = range;
    const popup = document.createElement("div");
    popup.id = POPUP_ID;
    popup.innerHTML = content;
    const { top, left } = calculatePopupPosition(range);
    popup.style.top = `${top}px`;
    popup.style.left = `${left}px`;
    document.body.appendChild(popup);
    currentPopup = popup;
    const closeBtn = popup.querySelector(".engeagle-close");
    if (closeBtn) {
      closeBtn.addEventListener("click", (e) => {
        e.stopPropagation();
        hidePopup();
      });
    }
    scrollHandler = () => {
      requestAnimationFrame(updatePopupPosition);
    };
    resizeHandler = () => {
      requestAnimationFrame(updatePopupPosition);
    };
    window.addEventListener("scroll", scrollHandler, true);
    window.addEventListener("resize", resizeHandler);
    setTimeout(() => {
      document.addEventListener("click", handleOutsideClick);
    }, 100);
  }
  function handleOutsideClick(e) {
    if (currentPopup && !currentPopup.contains(e.target)) {
      hidePopup();
      document.removeEventListener("click", handleOutsideClick);
    }
  }
  function showLoadingPopup(range) {
    const content = `
    <div class="engeagle-loading">
      \uC870\uD68C \uC911...
    </div>
  `;
    showPopup(range, content);
  }
  function showResultPopup(range, entry) {
    const posLabels = {
      "noun": "\uBA85\uC0AC",
      "verb": "\uB3D9\uC0AC",
      "adj": "\uD615\uC6A9\uC0AC",
      "adv": "\uBD80\uC0AC",
      "prep": "\uC804\uCE58\uC0AC",
      "conj": "\uC811\uC18D\uC0AC",
      "pron": "\uB300\uBA85\uC0AC",
      "interj": "\uAC10\uD0C4\uC0AC",
      "unknown": "",
      "": ""
    };
    const posLabel = posLabels[entry.pos] ?? entry.pos;
    const meanings = entry.meanings.join(", ");
    const content = `
    <button class="engeagle-close" title="\uB2EB\uAE30">&times;</button>
    <div class="engeagle-header">
      <span class="engeagle-word">${escapeHtml(entry.word)}</span>
      ${posLabel ? `<span class="engeagle-pos">${escapeHtml(posLabel)}</span>` : ""}
    </div>
    <div class="engeagle-meanings">${escapeHtml(meanings)}</div>
    ${entry.example ? `<div class="engeagle-example">${escapeHtml(entry.example)}</div>` : ""}
    <button class="engeagle-save-btn" data-word="${escapeHtml(entry.word)}" data-entry='${JSON.stringify(entry).replace(/'/g, "&#39;")}'>\u{1F4E5} \uB2E8\uC5B4\uC7A5\uC5D0 \uC800\uC7A5</button>
  `;
    showPopup(range, content);
    setTimeout(() => {
      const saveBtn = document.querySelector(".engeagle-save-btn");
      if (saveBtn) {
        saveBtn.addEventListener("click", async (e) => {
          e.stopPropagation();
          const btn = e.target;
          const entryData = JSON.parse(btn.dataset.entry || "{}");
          try {
            await chrome.runtime.sendMessage({
              type: "SAVE",
              entry: {
                word: entryData.word,
                lemma: entryData.lemma || entryData.word.toLowerCase(),
                pos: entryData.pos,
                meanings: entryData.meanings,
                example: entryData.example || "",
                source_url: window.location.href
              }
            });
            btn.textContent = "\u2705 \uC800\uC7A5\uB428";
            btn.disabled = true;
            btn.style.background = "#10b981";
            showToast("\uB2E8\uC5B4\uC7A5\uC5D0 \uC800\uC7A5\uB428");
          } catch (error) {
            console.error("[EngEagle] Save error:", error);
            btn.textContent = "\u274C \uC800\uC7A5 \uC2E4\uD328";
          }
        });
      }
    }, 50);
  }
  function showErrorPopup(range, message) {
    const content = `
    <button class="engeagle-close" title="\uB2EB\uAE30">&times;</button>
    <div class="engeagle-error">${escapeHtml(message)}</div>
  `;
    showPopup(range, content);
  }
  function escapeHtml(text) {
    const div = document.createElement("div");
    div.textContent = text;
    return div.innerHTML;
  }
  function getSelectedWord() {
    const selection = window.getSelection();
    if (!selection || selection.rangeCount === 0) {
      return null;
    }
    const range = selection.getRangeAt(0);
    let text = selection.toString().trim();
    if (text.includes(" ")) {
      text = text.split(/\s+/)[0];
    }
    if (!isEnglishWord(text)) {
      return null;
    }
    return { word: text, range };
  }
  var triggerConfig = {
    altDblclick: false,
    // Alt/Option + 더블클릭 = 번역
    shiftDblclick: true,
    // Shift + 더블클릭 = 번역 (기본)
    contextMenu: true
  };
  async function loadTriggerConfig() {
    try {
      const response = await chrome.runtime.sendMessage({ type: "TRIGGER_LOAD_CONFIG" });
      if (response.success && response.config) {
        triggerConfig = response.config;
        console.log("[EngEagle] Trigger config loaded:", triggerConfig);
      }
    } catch (error) {
      console.error("[EngEagle] Failed to load trigger config:", error);
    }
  }
  async function performTranslation(word, range) {
    const startTime = performance.now();
    showLoadingPopup(range);
    try {
      const response = await chrome.runtime.sendMessage({
        type: "LOOKUP",
        word,
        sourceUrl: window.location.href,
        saveToVocabulary: false
      });
      const elapsed = performance.now() - startTime;
      console.log(`[EngEagle] Lookup completed in ${elapsed.toFixed(1)}ms`);
      if (response.success && response.entry) {
        showResultPopup(range, response.entry);
      } else {
        showErrorPopup(range, "\uC0AC\uC804\uC5D0 \uC5C6\uB294 \uB2E8\uC5B4\uC785\uB2C8\uB2E4");
      }
    } catch (error) {
      console.error("[EngEagle] Lookup error:", error);
      showErrorPopup(range, "\uC870\uD68C \uC911 \uC624\uB958\uAC00 \uBC1C\uC0DD\uD588\uC2B5\uB2C8\uB2E4");
    }
  }
  async function handleDoubleClick(e) {
    const target = e.target;
    if (target.tagName === "INPUT" || target.tagName === "TEXTAREA" || target.isContentEditable) {
      return;
    }
    let shouldTranslate = false;
    if (e.altKey && triggerConfig.altDblclick) {
      shouldTranslate = true;
    } else if (e.shiftKey && triggerConfig.shiftDblclick) {
      shouldTranslate = true;
    }
    if (!shouldTranslate) return;
    const selected = getSelectedWord();
    if (!selected) {
      return;
    }
    await performTranslation(selected.word, selected.range);
  }
  function handleKeyDown(e) {
    if (e.key === "Escape" && currentPopup) {
      hidePopup();
      document.removeEventListener("click", handleOutsideClick);
    }
  }
  chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
    console.log("[EngEagle] Content script received message:", message.type);
    if (message.type === "SHOW_TRANSLATION") {
      console.log("[EngEagle] SHOW_TRANSLATION received:", message);
      const selection = window.getSelection();
      if (selection && selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        if (message.result && message.result.found && message.result.entry) {
          console.log("[EngEagle] Showing popup for:", message.result.entry);
          showResultPopup(range, message.result.entry);
        } else {
          console.log("[EngEagle] Word not found in dictionary");
          showErrorPopup(range, "\uC0AC\uC804\uC5D0 \uC5C6\uB294 \uB2E8\uC5B4\uC785\uB2C8\uB2E4");
        }
      } else {
        console.log("[EngEagle] No selection found");
      }
      sendResponse({ success: true });
    } else if (message.type === "TRIGGER_CONFIG_CHANGED") {
      triggerConfig = message.config;
      console.log("[EngEagle] Trigger config updated:", triggerConfig);
      sendResponse({ success: true });
    }
    return true;
  });
  document.addEventListener("dblclick", handleDoubleClick);
  document.addEventListener("keydown", handleKeyDown);
  loadTriggerConfig();
  console.log("[EngEagle] Content script loaded");
})();
