true&&(function polyfill() {
  const relList = document.createElement("link").relList;
  if (relList && relList.supports && relList.supports("modulepreload")) {
    return;
  }
  for (const link of document.querySelectorAll('link[rel="modulepreload"]')) {
    processPreload(link);
  }
  new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type !== "childList") {
        continue;
      }
      for (const node of mutation.addedNodes) {
        if (node.tagName === "LINK" && node.rel === "modulepreload")
          processPreload(node);
      }
    }
  }).observe(document, { childList: true, subtree: true });
  function getFetchOpts(link) {
    const fetchOpts = {};
    if (link.integrity) fetchOpts.integrity = link.integrity;
    if (link.referrerPolicy) fetchOpts.referrerPolicy = link.referrerPolicy;
    if (link.crossOrigin === "use-credentials")
      fetchOpts.credentials = "include";
    else if (link.crossOrigin === "anonymous") fetchOpts.credentials = "omit";
    else fetchOpts.credentials = "same-origin";
    return fetchOpts;
  }
  function processPreload(link) {
    if (link.ep)
      return;
    link.ep = true;
    const fetchOpts = getFetchOpts(link);
    fetch(link.href, fetchOpts);
  }
}());

const wordCountEl = document.getElementById("word-count");
const dictStatusEl = document.getElementById("dict-status");
const settingsToggle = document.getElementById("settings-toggle");
const settingsContent = document.getElementById("settings-content");
const deepLApiKeyInput = document.getElementById("deepl-api-key");
const btnToggleKey = document.getElementById("btn-toggle-key");
const btnTestDeepL = document.getElementById("btn-test-deepl");
const btnSaveDeepL = document.getElementById("btn-save-deepl");
const deepLStatus = document.getElementById("deepl-status");
const deepLUsage = document.getElementById("deepl-usage");
const usageCount = document.getElementById("usage-count");
const usageLimit = document.getElementById("usage-limit");
const searchInput = document.getElementById("search-input");
const searchClear = document.getElementById("search-clear");
const vocabBody = document.getElementById("vocab-body");
const emptyState = document.getElementById("empty-state");
const loadingState = document.getElementById("loading-state");
const undoToast = document.getElementById("undo-toast");
const undoMessage = undoToast.querySelector(".undo-message");
const btnUndo = document.getElementById("btn-undo");
const addModal = document.getElementById("add-modal");
const addForm = document.getElementById("add-form");
const btnAdd = document.getElementById("btn-add");
const btnCancel = document.getElementById("btn-cancel");
const btnExport = document.getElementById("btn-export");
const exportMenu = document.getElementById("export-menu");
const btnImport = document.getElementById("btn-import");
const fileImport = document.getElementById("file-import");
const triggerToggle = document.getElementById("trigger-toggle");
const triggerContent = document.getElementById("trigger-content");
const triggerAltDblclick = document.getElementById("trigger-alt-dblclick");
const triggerShiftDblclick = document.getElementById("trigger-shift-dblclick");
const triggerContextMenu = document.getElementById("trigger-context-menu");
const btnSaveTrigger = document.getElementById("btn-save-trigger");
const triggerStatus = document.getElementById("trigger-status");
let allEntries = [];
let filteredEntries = [];
let deletedEntry = null;
let undoTimeout = null;
let currentSort = { key: "created_at", asc: false };
const posLabels = {
  "noun": "명사",
  "verb": "동사",
  "adj": "형용사",
  "adv": "부사",
  "prep": "전치사",
  "conj": "접속사",
  "pron": "대명사",
  "interj": "감탄사",
  "other": "기타",
  "": "품사 선택"
};
const posOptions = [
  { value: "", label: "없음" },
  { value: "noun", label: "명사" },
  { value: "verb", label: "동사" },
  { value: "adj", label: "형용사" },
  { value: "adv", label: "부사" },
  { value: "prep", label: "전치사" },
  { value: "conj", label: "접속사" },
  { value: "pron", label: "대명사" },
  { value: "interj", label: "감탄사" },
  { value: "other", label: "기타" }
];
async function init() {
  await loadStatus();
  await loadVocabulary();
  await loadDeepLConfig();
  await loadTriggerConfig();
  setupEventListeners();
  setupDeepLEventListeners();
  setupTriggerEventListeners();
}
async function loadStatus() {
  try {
    const response = await chrome.runtime.sendMessage({ type: "STATUS" });
    if (response.success) {
      dictStatusEl.textContent = response.dictionaryLoaded ? "정상" : "오류";
      dictStatusEl.style.color = response.dictionaryLoaded ? "#3fb950" : "#f85149";
    }
  } catch (error) {
    console.error("Status load failed:", error);
    dictStatusEl.textContent = "오류";
    dictStatusEl.style.color = "#f85149";
  }
}
async function loadVocabulary() {
  loadingState.style.display = "block";
  emptyState.style.display = "none";
  vocabBody.innerHTML = "";
  try {
    const response = await chrome.runtime.sendMessage({ type: "GET_ALL" });
    if (response.success) {
      allEntries = response.entries;
      wordCountEl.textContent = allEntries.length.toString();
      filterAndRender();
    }
  } catch (error) {
    console.error("Vocabulary load failed:", error);
  } finally {
    loadingState.style.display = "none";
  }
}
function filterAndRender() {
  const query = searchInput.value.toLowerCase().trim();
  if (query) {
    filteredEntries = allEntries.filter(
      (entry) => entry.word.toLowerCase().startsWith(query) || entry.lemma.toLowerCase().startsWith(query) || entry.meanings.some((m) => m.includes(query))
    );
    searchClear.style.display = "block";
  } else {
    filteredEntries = [...allEntries];
    searchClear.style.display = "none";
  }
  sortEntries();
  renderTable();
}
function sortEntries() {
  const { key, asc } = currentSort;
  filteredEntries.sort((a, b) => {
    let valA;
    let valB;
    switch (key) {
      case "word":
        valA = a.word.toLowerCase();
        valB = b.word.toLowerCase();
        break;
      case "freq":
        valA = a.freq;
        valB = b.freq;
        break;
      case "created_at":
      default:
        valA = a.created_at;
        valB = b.created_at;
    }
    if (valA < valB) return asc ? -1 : 1;
    if (valA > valB) return asc ? 1 : -1;
    return 0;
  });
}
function renderTable() {
  vocabBody.innerHTML = "";
  if (filteredEntries.length === 0) {
    emptyState.style.display = "block";
    return;
  }
  emptyState.style.display = "none";
  for (const entry of filteredEntries) {
    const tr = document.createElement("tr");
    tr.dataset.id = entry.id;
    const posLabel = posLabels[entry.pos] || entry.pos || "품사 선택";
    const meanings = entry.meanings.join(", ");
    const date = formatDate(entry.created_at);
    const posClass = entry.pos ? "word-pos" : "word-pos word-pos-empty";
    const favoriteClass = entry.favorite ? "btn-favorite active" : "btn-favorite";
    const favoriteIcon = entry.favorite ? "⭐" : "☆";
    tr.innerHTML = `
      <td class="col-fav">
        <button class="${favoriteClass}" title="즐겨찾기 토글">${favoriteIcon}</button>
      </td>
      <td class="col-word">
        <div class="word-cell">
          <span class="word-text">${escapeHtml(entry.word)}</span>
          <span class="${posClass}" data-pos="${escapeHtml(entry.pos)}" title="클릭하여 품사 편집">${escapeHtml(posLabel)}</span>
        </div>
      </td>
      <td class="col-meaning">
        <span class="meaning-cell">${escapeHtml(meanings)}</span>
      </td>
      <td class="col-example">
        <span class="example-cell">${escapeHtml(entry.example)}</span>
      </td>
      <td class="col-freq">
        <span class="freq-cell">${entry.freq}</span>
      </td>
      <td class="col-date">
        <span class="date-cell">${date}</span>
      </td>
      <td class="col-actions">
        <button class="btn btn-danger btn-delete" title="삭제">🗑</button>
      </td>
    `;
    vocabBody.appendChild(tr);
  }
}
function formatDate(timestamp) {
  const date = new Date(timestamp * 1e3);
  const year = date.getFullYear();
  const month = String(date.getMonth() + 1).padStart(2, "0");
  const day = String(date.getDate()).padStart(2, "0");
  return `${year}.${month}.${day}`;
}
function escapeHtml(text) {
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}
function setupEventListeners() {
  searchInput.addEventListener("input", () => {
    filterAndRender();
  });
  searchClear.addEventListener("click", () => {
    searchInput.value = "";
    filterAndRender();
  });
  document.querySelectorAll("th[data-sort]").forEach((th) => {
    th.addEventListener("click", () => {
      const key = th.getAttribute("data-sort");
      if (currentSort.key === key) {
        currentSort.asc = !currentSort.asc;
      } else {
        currentSort.key = key;
        currentSort.asc = key === "word";
      }
      document.querySelectorAll("th[data-sort]").forEach((el) => {
        el.classList.remove("sort-asc", "sort-desc");
      });
      th.classList.add(currentSort.asc ? "sort-asc" : "sort-desc");
      filterAndRender();
    });
  });
  vocabBody.addEventListener("click", async (e) => {
    const target = e.target;
    if (target.classList.contains("btn-delete")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      deleteEntry(id);
    } else if (target.classList.contains("btn-favorite")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      await toggleFavorite(id, target);
    } else if (target.classList.contains("word-pos")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      showPosEditor(target, id);
    } else if (target.classList.contains("word-text")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      showInlineEditor(target, id, "word");
    } else if (target.classList.contains("meaning-cell")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      showInlineEditor(target, id, "meanings");
    } else if (target.classList.contains("example-cell")) {
      const tr = target.closest("tr");
      const id = tr.dataset.id;
      showInlineEditor(target, id, "example");
    }
  });
  btnUndo.addEventListener("click", undoDelete);
  btnAdd.addEventListener("click", () => {
    addModal.style.display = "flex";
    document.getElementById("add-word").focus();
  });
  btnCancel.addEventListener("click", () => {
    addModal.style.display = "none";
    addForm.reset();
  });
  addModal.querySelector(".modal-backdrop")?.addEventListener("click", () => {
    addModal.style.display = "none";
    addForm.reset();
  });
  addForm.addEventListener("submit", async (e) => {
    e.preventDefault();
    await addNewWord();
  });
  btnExport.addEventListener("click", (e) => {
    e.stopPropagation();
    exportMenu.classList.toggle("show");
  });
  document.addEventListener("click", () => {
    exportMenu.classList.remove("show");
  });
  exportMenu.querySelectorAll("button").forEach((btn) => {
    btn.addEventListener("click", () => {
      const format = btn.dataset.format;
      exportData(format);
    });
  });
  btnImport.addEventListener("click", () => {
    fileImport.click();
  });
  fileImport.addEventListener("change", async () => {
    const file = fileImport.files?.[0];
    if (file) {
      await importData(file);
      fileImport.value = "";
    }
  });
  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape" && addModal.style.display === "flex") {
      addModal.style.display = "none";
      addForm.reset();
    }
  });
}
async function deleteEntry(id) {
  const entry = allEntries.find((e) => e.id === id);
  if (!entry) return;
  try {
    const response = await chrome.runtime.sendMessage({ type: "DELETE", id });
    if (response.success) {
      allEntries = allEntries.filter((e) => e.id !== id);
      wordCountEl.textContent = allEntries.length.toString();
      filterAndRender();
      showUndoToast(entry);
    }
  } catch (error) {
    console.error("Delete failed:", error);
  }
}
async function toggleFavorite(id, button) {
  try {
    const response = await chrome.runtime.sendMessage({ type: "TOGGLE_FAVORITE", id });
    if (response.success && response.entry) {
      const entryIndex = allEntries.findIndex((e) => e.id === id);
      if (entryIndex !== -1) {
        allEntries[entryIndex] = response.entry;
      }
      if (response.entry.favorite) {
        button.classList.add("active");
        button.textContent = "⭐";
      } else {
        button.classList.remove("active");
        button.textContent = "☆";
      }
    }
  } catch (error) {
    console.error("Toggle favorite failed:", error);
  }
}
function showUndoToast(entry) {
  if (undoTimeout) {
    clearTimeout(undoTimeout);
  }
  deletedEntry = entry;
  undoMessage.textContent = `"${entry.word}" 삭제됨`;
  undoToast.style.display = "flex";
  undoTimeout = window.setTimeout(() => {
    hideUndoToast();
  }, 5e3);
}
function hideUndoToast() {
  undoToast.style.display = "none";
  deletedEntry = null;
  if (undoTimeout) {
    clearTimeout(undoTimeout);
    undoTimeout = null;
  }
}
async function undoDelete() {
  if (!deletedEntry) return;
  try {
    const response = await chrome.runtime.sendMessage({
      type: "RESTORE",
      entry: deletedEntry
    });
    if (response.success) {
      allEntries.push(deletedEntry);
      wordCountEl.textContent = allEntries.length.toString();
      filterAndRender();
      hideUndoToast();
    }
  } catch (error) {
    console.error("Undo failed:", error);
  }
}
async function addNewWord() {
  const word = document.getElementById("add-word").value.trim();
  const pos = document.getElementById("add-pos").value;
  const meaningsStr = document.getElementById("add-meanings").value.trim();
  const example = document.getElementById("add-example").value.trim();
  if (!word || !meaningsStr) return;
  const meanings = meaningsStr.split(",").map((m) => m.trim()).filter((m) => m);
  try {
    const response = await chrome.runtime.sendMessage({
      type: "SAVE",
      entry: {
        word,
        lemma: word.toLowerCase(),
        pos,
        meanings,
        example,
        source_url: ""
      }
    });
    if (response.success) {
      addModal.style.display = "none";
      addForm.reset();
      await loadVocabulary();
    }
  } catch (error) {
    console.error("Add failed:", error);
  }
}
async function exportData(format) {
  try {
    const response = await chrome.runtime.sendMessage({
      type: "EXPORT",
      format
    });
    if (response.success) {
      const blob = new Blob([response.data], {
        type: format === "json" ? "application/json" : "text/csv"
      });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `engeagle_vocabulary.${format}`;
      a.click();
      URL.revokeObjectURL(url);
    }
  } catch (error) {
    console.error("Export failed:", error);
  }
}
async function importData(file) {
  try {
    const text = await file.text();
    const response = await chrome.runtime.sendMessage({
      type: "IMPORT",
      data: text
    });
    if (response.success) {
      alert(`${response.imported}개의 단어를 가져왔습니다.`);
      await loadVocabulary();
    }
  } catch (error) {
    console.error("Import failed:", error);
    alert("가져오기 실패: 올바른 JSON 파일인지 확인하세요.");
  }
}
async function loadDeepLConfig() {
  try {
    const response = await chrome.runtime.sendMessage({ type: "DEEPL_LOAD_CONFIG" });
    if (response.success && response.config) {
      deepLApiKeyInput.value = response.config.apiKey || "";
    }
  } catch (error) {
    console.error("DeepL config load failed:", error);
  }
}
function setupDeepLEventListeners() {
  settingsToggle.addEventListener("click", () => {
    const isOpen = settingsContent.style.display !== "none";
    settingsContent.style.display = isOpen ? "none" : "block";
    settingsToggle.classList.toggle("open", !isOpen);
  });
  btnToggleKey.addEventListener("click", () => {
    const isPassword = deepLApiKeyInput.type === "password";
    deepLApiKeyInput.type = isPassword ? "text" : "password";
    btnToggleKey.textContent = isPassword ? "🙈" : "👁";
  });
  btnTestDeepL.addEventListener("click", testDeepLConnection);
  btnSaveDeepL.addEventListener("click", saveDeepLConfig);
}
async function testDeepLConnection() {
  const apiKey = deepLApiKeyInput.value.trim();
  if (!apiKey) {
    showDeepLStatus("API 키를 입력해주세요.", "error");
    return;
  }
  showDeepLStatus("연결 테스트 중...", "loading");
  btnTestDeepL.disabled = true;
  try {
    const response = await chrome.runtime.sendMessage({
      type: "DEEPL_TEST",
      config: { apiKey, useFreeApi: apiKey.endsWith(":fx") }
    });
    if (response.success) {
      showDeepLStatus("✅ " + response.message, "success");
      if (response.usage) {
        deepLUsage.style.display = "flex";
        usageCount.textContent = response.usage.character_count.toLocaleString();
        usageLimit.textContent = response.usage.character_limit.toLocaleString();
      }
    } else {
      showDeepLStatus("❌ " + response.message, "error");
      deepLUsage.style.display = "none";
    }
  } catch (error) {
    showDeepLStatus("❌ 테스트 실패", "error");
    console.error("DeepL test failed:", error);
  } finally {
    btnTestDeepL.disabled = false;
  }
}
async function saveDeepLConfig() {
  const apiKey = deepLApiKeyInput.value.trim();
  btnSaveDeepL.disabled = true;
  try {
    const response = await chrome.runtime.sendMessage({
      type: "DEEPL_SAVE_CONFIG",
      config: { apiKey, useFreeApi: apiKey.endsWith(":fx") }
    });
    if (response.success) {
      showDeepLStatus("✅ 설정이 저장되었습니다.", "success");
    } else {
      showDeepLStatus("❌ 저장 실패", "error");
    }
  } catch (error) {
    showDeepLStatus("❌ 저장 실패", "error");
    console.error("DeepL save failed:", error);
  } finally {
    btnSaveDeepL.disabled = false;
  }
}
function showDeepLStatus(message, type) {
  deepLStatus.textContent = message;
  deepLStatus.className = "deepl-status " + type;
  deepLStatus.style.display = "block";
}
function showPosEditor(target, id) {
  const existingEditor = document.querySelector(".pos-editor");
  if (existingEditor) {
    existingEditor.remove();
  }
  const currentPos = target.dataset.pos || "";
  const select = document.createElement("select");
  select.className = "pos-editor";
  for (const option of posOptions) {
    const opt = document.createElement("option");
    opt.value = option.value;
    opt.textContent = option.label;
    if (option.value === currentPos) {
      opt.selected = true;
    }
    select.appendChild(opt);
  }
  const rect = target.getBoundingClientRect();
  select.style.position = "fixed";
  select.style.top = `${rect.top}px`;
  select.style.left = `${rect.left}px`;
  select.style.zIndex = "1000";
  select.addEventListener("change", async () => {
    const newPos = select.value;
    await updatePos(id, newPos);
    select.remove();
  });
  select.addEventListener("blur", () => {
    setTimeout(() => select.remove(), 100);
  });
  select.addEventListener("keydown", (e) => {
    if (e.key === "Escape") {
      select.remove();
    }
  });
  document.body.appendChild(select);
  select.focus();
}
async function updatePos(id, newPos) {
  try {
    const response = await chrome.runtime.sendMessage({
      type: "UPDATE",
      id,
      updates: { pos: newPos }
    });
    if (response.success) {
      const entry = allEntries.find((e) => e.id === id);
      if (entry) {
        entry.pos = newPos;
      }
      filterAndRender();
    }
  } catch (error) {
    console.error("Update failed:", error);
  }
}
function showInlineEditor(target, id, field) {
  if (target.querySelector("input")) return;
  const entry = allEntries.find((e) => e.id === id);
  if (!entry) return;
  let currentValue = "";
  if (field === "word") {
    currentValue = entry.word;
  } else if (field === "meanings") {
    currentValue = entry.meanings.join(", ");
  } else if (field === "example") {
    currentValue = entry.example;
  }
  const originalText = target.textContent || "";
  const originalWidth = target.offsetWidth;
  const input = document.createElement("input");
  input.type = "text";
  input.value = currentValue;
  input.className = "inline-editor";
  input.style.width = `${Math.max(originalWidth, 100)}px`;
  target.textContent = "";
  target.appendChild(input);
  input.focus();
  input.select();
  const saveEdit = async () => {
    const newValue = input.value.trim();
    if (!newValue && field !== "example") {
      target.textContent = originalText;
      return;
    }
    if (newValue === currentValue) {
      target.textContent = originalText;
      return;
    }
    try {
      let updates = {};
      if (field === "word") {
        updates = { word: newValue, lemma: newValue.toLowerCase() };
      } else if (field === "meanings") {
        updates = { meanings: newValue.split(",").map((m) => m.trim()).filter((m) => m) };
      } else if (field === "example") {
        updates = { example: newValue };
      }
      const response = await chrome.runtime.sendMessage({
        type: "UPDATE",
        id,
        updates
      });
      if (response.success) {
        const entryIdx = allEntries.findIndex((e) => e.id === id);
        if (entryIdx !== -1) {
          if (field === "word") {
            allEntries[entryIdx].word = newValue;
            allEntries[entryIdx].lemma = newValue.toLowerCase();
          } else if (field === "meanings") {
            allEntries[entryIdx].meanings = newValue.split(",").map((m) => m.trim()).filter((m) => m);
          } else if (field === "example") {
            allEntries[entryIdx].example = newValue;
          }
        }
        filterAndRender();
      } else {
        target.textContent = originalText;
      }
    } catch (error) {
      console.error("Update failed:", error);
      target.textContent = originalText;
    }
  };
  input.addEventListener("blur", saveEdit);
  input.addEventListener("keydown", (e) => {
    if (e.key === "Enter") {
      e.preventDefault();
      input.blur();
    } else if (e.key === "Escape") {
      target.textContent = originalText;
    }
  });
}
async function loadTriggerConfig() {
  try {
    const response = await chrome.runtime.sendMessage({ type: "TRIGGER_LOAD_CONFIG" });
    if (response.success && response.config) {
      triggerAltDblclick.checked = response.config.altDblclick ?? false;
      triggerShiftDblclick.checked = response.config.shiftDblclick ?? true;
      triggerShiftDblclick.checked = response.config.shiftDblclick ?? false;
      triggerContextMenu.checked = response.config.contextMenu ?? true;
    }
  } catch (error) {
    console.error("Trigger config load failed:", error);
  }
}
function setupTriggerEventListeners() {
  triggerToggle.addEventListener("click", () => {
    const isOpen = triggerContent.style.display !== "none";
    triggerContent.style.display = isOpen ? "none" : "block";
    triggerToggle.classList.toggle("open", !isOpen);
  });
  btnSaveTrigger.addEventListener("click", saveTriggerConfig);
}
async function saveTriggerConfig() {
  const config = {
    altDblclick: triggerAltDblclick.checked,
    shiftDblclick: triggerShiftDblclick.checked,
    contextMenu: triggerContextMenu.checked
  };
  btnSaveTrigger.disabled = true;
  try {
    const response = await chrome.runtime.sendMessage({
      type: "TRIGGER_SAVE_CONFIG",
      config
    });
    if (response.success) {
      showTriggerStatus("✅ 설정이 저장되었습니다.", "success");
    } else {
      showTriggerStatus("❌ 저장 실패", "error");
    }
  } catch (error) {
    showTriggerStatus("❌ 저장 실패", "error");
    console.error("Trigger save failed:", error);
  } finally {
    btnSaveTrigger.disabled = false;
  }
}
function showTriggerStatus(message, type) {
  triggerStatus.textContent = message;
  triggerStatus.className = "deepl-status " + type;
  triggerStatus.style.display = "block";
  setTimeout(() => {
    triggerStatus.style.display = "none";
  }, 3e3);
}
const quizToggle = document.getElementById("quiz-toggle");
const quizContent = document.getElementById("quiz-content");
const quizStart = document.getElementById("quiz-start");
const quizPlay = document.getElementById("quiz-play");
const quizResult = document.getElementById("quiz-result");
let quizEntries = [];
let currentQuizIndex = 0;
let quizStats = { favorited: 0, deleted: 0, reviewed: 0 };
function initQuiz() {
  quizToggle.addEventListener("click", () => {
    const isOpen = quizContent.style.display !== "none";
    quizContent.style.display = isOpen ? "none" : "block";
    quizToggle.classList.toggle("open", !isOpen);
    if (!isOpen) updateQuizStats();
  });
  document.getElementById("btn-start-quiz")?.addEventListener("click", startQuiz);
  document.getElementById("btn-show-answer")?.addEventListener("click", showAnswer);
  document.getElementById("btn-quiz-favorite")?.addEventListener("click", toggleQuizFavorite);
  document.getElementById("btn-quiz-delete")?.addEventListener("click", deleteQuizWord);
  document.getElementById("btn-quiz-next")?.addEventListener("click", nextQuizWord);
  document.getElementById("btn-quiz-end")?.addEventListener("click", endQuiz);
  document.getElementById("btn-quiz-restart")?.addEventListener("click", restartQuiz);
}
function updateQuizStats() {
  const total = allEntries.length;
  const favorites = allEntries.filter((e) => e.favorite).length;
  const totalEl = document.getElementById("quiz-total-count");
  const favEl = document.getElementById("quiz-favorite-count");
  if (totalEl) totalEl.textContent = total.toString();
  if (favEl) favEl.textContent = favorites.toString();
}
function startQuiz() {
  const mode = document.querySelector('input[name="quiz-mode"]:checked')?.value || "all";
  if (mode === "favorites") {
    quizEntries = allEntries.filter((e) => e.favorite);
  } else {
    quizEntries = [...allEntries];
  }
  if (quizEntries.length === 0) {
    alert(mode === "favorites" ? "즐겨찾기된 단어가 없습니다." : "단어장이 비어있습니다.");
    return;
  }
  quizEntries = shuffleArray(quizEntries);
  currentQuizIndex = 0;
  quizStats = { favorited: 0, deleted: 0, reviewed: 0 };
  quizStart.style.display = "none";
  quizPlay.style.display = "block";
  quizResult.style.display = "none";
  showQuizWord();
}
function shuffleArray(array) {
  const shuffled = [...array];
  for (let i = shuffled.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]];
  }
  return shuffled;
}
function showQuizWord() {
  const entry = quizEntries[currentQuizIndex];
  if (!entry) return;
  const currentEl = document.getElementById("quiz-current");
  const totalEl = document.getElementById("quiz-total");
  if (currentEl) currentEl.textContent = (currentQuizIndex + 1).toString();
  if (totalEl) totalEl.textContent = quizEntries.length.toString();
  const wordEl = document.getElementById("quiz-word");
  const posEl = document.getElementById("quiz-pos");
  const meaningsEl = document.getElementById("quiz-meanings");
  const exampleEl = document.getElementById("quiz-example");
  if (wordEl) wordEl.textContent = entry.word;
  if (posEl) posEl.textContent = getPosLabel(entry.pos);
  if (meaningsEl) meaningsEl.textContent = entry.meanings.join(", ");
  if (exampleEl) {
    exampleEl.textContent = entry.example || "";
    exampleEl.style.display = entry.example ? "block" : "none";
  }
  const answerEl = document.getElementById("quiz-answer");
  const actionsEl = document.getElementById("quiz-actions");
  const showBtn = document.getElementById("btn-show-answer");
  if (answerEl) answerEl.style.display = "none";
  if (actionsEl) actionsEl.style.display = "none";
  if (showBtn) showBtn.style.display = "block";
  updateFavoriteButton(entry.favorite || false);
}
function getPosLabel(pos) {
  const labels = {
    "noun": "명사",
    "verb": "동사",
    "adj": "형용사",
    "adv": "부사",
    "prep": "전치사",
    "conj": "접속사",
    "pron": "대명사",
    "interj": "감탄사"
  };
  return labels[pos] || pos || "";
}
function showAnswer() {
  const answerEl = document.getElementById("quiz-answer");
  const actionsEl = document.getElementById("quiz-actions");
  const showBtn = document.getElementById("btn-show-answer");
  if (answerEl) answerEl.style.display = "block";
  if (actionsEl) actionsEl.style.display = "flex";
  if (showBtn) showBtn.style.display = "none";
  quizStats.reviewed++;
}
async function toggleQuizFavorite() {
  const entry = quizEntries[currentQuizIndex];
  if (!entry) return;
  const newFavorite = !entry.favorite;
  entry.favorite = newFavorite;
  try {
    await chrome.runtime.sendMessage({
      type: "UPDATE",
      id: entry.id,
      updates: { favorite: newFavorite }
    });
    updateFavoriteButton(newFavorite);
    if (newFavorite) {
      quizStats.favorited++;
    } else {
      quizStats.favorited = Math.max(0, quizStats.favorited - 1);
    }
    const idx = allEntries.findIndex((e) => e.id === entry.id);
    if (idx !== -1) {
      allEntries[idx].favorite = newFavorite;
    }
  } catch (error) {
    console.error("Failed to update favorite:", error);
  }
}
function updateFavoriteButton(isFavorite) {
  const btn = document.getElementById("btn-quiz-favorite");
  if (btn) {
    btn.innerHTML = isFavorite ? '<span class="btn-icon">✅</span> 즐겨찾기됨' : '<span class="btn-icon">⭐</span> 즐겨찾기';
    btn.classList.toggle("active", isFavorite);
  }
}
async function deleteQuizWord() {
  const entry = quizEntries[currentQuizIndex];
  if (!entry) return;
  if (!confirm(`"${entry.word}"를 단어장에서 삭제하시겠습니까?`)) return;
  try {
    await chrome.runtime.sendMessage({
      type: "DELETE",
      id: entry.id
    });
    quizStats.deleted++;
    allEntries = allEntries.filter((e) => e.id !== entry.id);
    quizEntries.splice(currentQuizIndex, 1);
    if (quizEntries.length === 0) {
      endQuiz();
    } else {
      if (currentQuizIndex >= quizEntries.length) {
        currentQuizIndex = quizEntries.length - 1;
      }
      showQuizWord();
    }
  } catch (error) {
    console.error("Failed to delete word:", error);
  }
}
function nextQuizWord() {
  currentQuizIndex++;
  if (currentQuizIndex >= quizEntries.length) {
    endQuiz();
  } else {
    showQuizWord();
  }
}
function endQuiz() {
  quizPlay.style.display = "none";
  quizResult.style.display = "block";
  const reviewedEl = document.getElementById("quiz-reviewed-count");
  const favoritedEl = document.getElementById("result-favorited");
  const deletedEl = document.getElementById("result-deleted");
  if (reviewedEl) reviewedEl.textContent = quizStats.reviewed.toString();
  if (favoritedEl) favoritedEl.textContent = quizStats.favorited.toString();
  if (deletedEl) deletedEl.textContent = quizStats.deleted.toString();
  loadVocabulary();
}
function restartQuiz() {
  quizResult.style.display = "none";
  quizStart.style.display = "block";
  updateQuizStats();
}
initQuiz();
init();
